/**
 * DevExtreme (ui/grid_core/ui.grid_core.editing.js)
 * Version: 17.1.6
 * Build date: Tue Sep 05 2017
 *
 * Copyright (c) 2012 - 2017 Developer Express Inc. ALL RIGHTS RESERVED
 * Read about DevExtreme licensing here: https://js.devexpress.com/Licensing/
 */
"use strict";
var $ = require("../../core/renderer"),
    Guid = require("../../core/guid"),
    commonUtils = require("../../core/utils/common"),
    typeUtils = require("../../core/utils/type"),
    deepExtendArraySafe = require("../../core/utils/object").deepExtendArraySafe,
    extend = require("../../core/utils/extend").extend,
    modules = require("./ui.grid_core.modules"),
    clickEvent = require("../../events/click"),
    gridCoreUtils = require("./ui.grid_core.utils"),
    getIndexByKey = gridCoreUtils.getIndexByKey,
    eventUtils = require("../../events/utils"),
    addNamespace = eventUtils.addNamespace,
    dialog = require("../dialog"),
    messageLocalization = require("../../localization/message"),
    Button = require("../button"),
    Popup = require("../popup"),
    errors = require("../widget/ui.errors"),
    devices = require("../../core/devices"),
    Form = require("../form"),
    holdEvent = require("../../events/hold"),
    when = require("../../integration/jquery/deferred").when;
var EDIT_FORM_CLASS = "edit-form",
    EDIT_FORM_ITEM_CLASS = "edit-form-item",
    FOCUS_OVERLAY_CLASS = "focus-overlay",
    READONLY_CLASS = "readonly",
    EDIT_POPUP_CLASS = "edit-popup",
    FORM_BUTTONS_CONTAINER_CLASS = "form-buttons-container",
    ADD_ROW_BUTTON_CLASS = "addrow-button",
    LINK_CLASS = "dx-link",
    EDITOR_CELL_CLASS = "dx-editor-cell",
    ROW_SELECTED = "dx-selection",
    EDIT_ROW = "dx-edit-row",
    EDIT_BUTTON_CLASS = "dx-edit-button",
    BUTTON_CLASS = "dx-button",
    INSERT_INDEX = "__DX_INSERT_INDEX__",
    ROW_CLASS = "dx-row",
    ROW_REMOVED = "dx-row-removed",
    ROW_INSERTED = "dx-row-inserted",
    ROW_MODIFIED = "dx-row-modified",
    CELL_MODIFIED = "dx-cell-modified",
    CELL_HIGHLIGHT_OUTLINE = "dx-highlight-outline",
    EDITING_NAMESPACE = "dxDataGridEditing",
    DATA_ROW_CLASS = "dx-data-row",
    CELL_FOCUS_DISABLED_CLASS = "dx-cell-focus-disabled",
    EDITORS_INPUT_SELECTOR = "input:not([type='hidden'])",
    FOCUSABLE_ELEMENT_SELECTOR = "[tabindex], " + EDITORS_INPUT_SELECTOR,
    EDIT_MODE_BATCH = "batch",
    EDIT_MODE_ROW = "row",
    EDIT_MODE_CELL = "cell",
    EDIT_MODE_FORM = "form",
    EDIT_MODE_POPUP = "popup",
    DATA_EDIT_DATA_INSERT_TYPE = "insert",
    DATA_EDIT_DATA_UPDATE_TYPE = "update",
    DATA_EDIT_DATA_REMOVE_TYPE = "remove",
    POINTER_EVENTS_NONE_CLASS = "dx-pointer-events-none",
    POINTER_EVENTS_TARGET_CLASS = "dx-pointer-events-target",
    EDIT_MODES = [EDIT_MODE_BATCH, EDIT_MODE_ROW, EDIT_MODE_CELL, EDIT_MODE_FORM, EDIT_MODE_POPUP],
    ROW_BASED_MODES = [EDIT_MODE_ROW, EDIT_MODE_FORM, EDIT_MODE_POPUP],
    CELL_BASED_MODES = [EDIT_MODE_BATCH, EDIT_MODE_CELL],
    MODES_WITH_DELAYED_FOCUS = [EDIT_MODE_ROW, EDIT_MODE_FORM];
var getEditMode = function(that) {
    var editMode = that.option("editing.mode");
    if (EDIT_MODES.indexOf(editMode) !== -1) {
        return editMode
    }
    return EDIT_MODE_ROW
};
var isRowEditMode = function(that) {
    var editMode = getEditMode(that);
    return ROW_BASED_MODES.indexOf(editMode) !== -1
};
var EditingController = modules.ViewController.inherit(function() {
    var getDefaultEditorTemplate = function(that) {
        return function(container, options) {
            var $editor = $("<div/>").appendTo(container);
            that.getController("editorFactory").createEditor($editor, extend({}, options.column, {
                value: options.value,
                setValue: options.setValue,
                row: options.row,
                parentType: "dataRow",
                width: null,
                readOnly: !options.setValue,
                isOnForm: options.isOnForm,
                id: options.id,
                updateValueImmediately: isRowEditMode(that)
            }))
        }
    };
    return {
        init: function() {
            var that = this;
            that._editRowIndex = -1;
            that._editData = [];
            that._editColumnIndex = -1;
            that._columnsController = that.getController("columns");
            that._dataController = that.getController("data");
            that._rowsView = that.getView("rowsView");
            if (!that._dataChangedHandler) {
                that._dataChangedHandler = that._handleDataChanged.bind(that);
                that._dataController.changed.add(that._dataChangedHandler)
            }
            if (!that._saveEditorHandler) {
                that.createAction("onInitNewRow", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowInserting", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowInserted", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onEditingStart", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowUpdating", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowUpdated", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowRemoving", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that.createAction("onRowRemoved", {
                    excludeValidators: ["disabled", "readOnly"]
                });
                that._saveEditorHandler = that.createAction(function(e) {
                    var isEditorPopup, isDomElement, isFocusOverlay, isAddRowButton, isCellEditMode, $target, event = e.jQueryEvent;
                    if (!isRowEditMode(that) && !that._editCellInProgress) {
                        $target = $(event.target);
                        isEditorPopup = $target.closest(".dx-dropdowneditor-overlay").length;
                        isDomElement = $target.closest(document).length;
                        isAddRowButton = $target.closest("." + that.addWidgetPrefix(ADD_ROW_BUTTON_CLASS)).length;
                        isFocusOverlay = $target.hasClass(that.addWidgetPrefix(FOCUS_OVERLAY_CLASS));
                        isCellEditMode = getEditMode(that) === EDIT_MODE_CELL;
                        if (!isEditorPopup && !isFocusOverlay && !(isAddRowButton && isCellEditMode && that.isEditing()) && isDomElement) {
                            that._closeEditItem.bind(that)($target)
                        }
                    }
                });
                $(document).on(clickEvent.name, that._saveEditorHandler)
            }
            that._updateEditColumn();
            that._updateEditButtons()
        },
        _closeEditItem: function($targetElement) {
            var isDataRow = $targetElement.closest("." + DATA_ROW_CLASS).length,
                $targetCell = $targetElement.closest("." + ROW_CLASS + "> td"),
                columnIndex = $targetCell[0] && $targetCell[0].cellIndex,
                rowIndex = this.getView("rowsView").getRowIndex($targetCell.parent()),
                visibleColumns = this._columnsController.getVisibleColumns(),
                allowEditing = visibleColumns[columnIndex] && visibleColumns[columnIndex].allowEditing;
            if (this.isEditing() && (!isDataRow || isDataRow && !allowEditing && !this.isEditCell(rowIndex, columnIndex))) {
                this.closeEditCell()
            }
        },
        _handleDataChanged: function(args) {
            if ("standard" === this.option("scrolling.mode")) {
                this.resetRowAndPageIndices()
            }
            if ("prepend" === args.changeType) {
                $.each(this._editData, function(_, editData) {
                    editData.rowIndex += args.items.length;
                    if (editData.type === DATA_EDIT_DATA_INSERT_TYPE) {
                        editData.key.rowIndex += args.items.length;
                        editData.key.dataRowIndex += args.items.filter(function(item) {
                            return "data" === item.rowType
                        }).length
                    }
                })
            }
        },
        isRowEditMode: function() {
            return isRowEditMode(this)
        },
        getEditMode: function() {
            return getEditMode(this)
        },
        getFirstEditableColumnIndex: function() {
            var columnIndex, columnsController = this.getController("columns");
            if (getEditMode(this) === EDIT_MODE_FORM && this._firstFormItem) {
                columnIndex = this._firstFormItem.column.index
            } else {
                var visibleColumns = columnsController.getVisibleColumns();
                $.each(visibleColumns, function(index, column) {
                    if (column.allowEditing) {
                        columnIndex = index;
                        return false
                    }
                })
            }
            return columnIndex
        },
        getFirstEditableCellInRow: function(rowIndex) {
            return this.getView("rowsView").getCellElement(rowIndex ? rowIndex : 0, this.getFirstEditableColumnIndex())
        },
        getFocusedCellInRow: function(rowIndex) {
            return this.getFirstEditableCellInRow(rowIndex)
        },
        getIndexByKey: function(key, items) {
            return getIndexByKey(key, items)
        },
        hasChanges: function() {
            var that = this,
                result = false;
            for (var i = 0; i < that._editData.length; i++) {
                if (that._editData[i].type) {
                    result = true;
                    break
                }
            }
            return result
        },
        dispose: function() {
            this.callBase();
            clearTimeout(this._inputFocusTimeoutID);
            $(document).off(clickEvent.name, this._saveEditorHandler)
        },
        optionChanged: function(args) {
            if ("editing" === args.name) {
                this.init();
                args.handled = true
            } else {
                this.callBase(args)
            }
        },
        publicMethods: function() {
            return ["insertRow", "addRow", "removeRow", "deleteRow", "undeleteRow", "editRow", "editCell", "closeEditCell", "saveEditData", "cancelEditData", "hasEditData"]
        },
        refresh: function() {
            if (getEditMode(this) === EDIT_MODE_CELL) {
                return
            }
            if (getEditMode(this) !== EDIT_MODE_BATCH) {
                this.init()
            } else {
                this._editRowIndex = -1;
                this._editColumnIndex = -1
            }
        },
        isEditing: function() {
            return this._editRowIndex > -1
        },
        isEditRow: function(rowIndex) {
            var editMode = getEditMode(this);
            return this._getVisibleEditRowIndex() === rowIndex && ROW_BASED_MODES.indexOf(editMode) !== -1
        },
        getEditRowKey: function() {
            var items = this._dataController.items(),
                item = items[this._getVisibleEditRowIndex()];
            return item && item.key
        },
        getEditFormRowIndex: function() {
            var editMode = getEditMode(this);
            return editMode === EDIT_MODE_FORM || editMode === EDIT_MODE_POPUP ? this._getVisibleEditRowIndex() : -1
        },
        isEditCell: function(rowIndex, columnIndex) {
            return this._getVisibleEditRowIndex() === rowIndex && this._editColumnIndex === columnIndex
        },
        getPopupContent: function() {
            var editMode = getEditMode(this),
                popupVisible = this._editPopup && this._editPopup.option("visible");
            if (editMode === EDIT_MODE_POPUP && popupVisible) {
                return this._editPopup.content()
            }
        },
        getEditForm: function() {
            return this._editForm
        },
        _needInsertItem: function(editData, changeType) {
            var that = this,
                dataSource = that._dataController.dataSource(),
                scrollingMode = that.option("scrolling.mode"),
                pageIndex = dataSource.pageIndex(),
                beginPageIndex = dataSource.beginPageIndex ? dataSource.beginPageIndex() : pageIndex,
                endPageIndex = dataSource.endPageIndex ? dataSource.endPageIndex() : pageIndex;
            if ("standard" !== scrollingMode) {
                switch (changeType) {
                    case "append":
                        return editData.key.pageIndex === endPageIndex;
                    case "prepend":
                        return editData.key.pageIndex === beginPageIndex;
                    case "refresh":
                        editData.key.rowIndex = 0;
                        editData.key.dataRowIndex = 0;
                        editData.key.pageIndex = 0;
                        break;
                    default:
                        return editData.key.pageIndex >= beginPageIndex && editData.key.pageIndex <= endPageIndex
                }
            }
            return editData.key.pageIndex === pageIndex
        },
        _generateNewItem: function(key) {
            var item = {
                key: key
            };
            if (key && key[INSERT_INDEX]) {
                item[INSERT_INDEX] = key[INSERT_INDEX]
            }
            return item
        },
        processItems: function(items, changeType) {
            var i, key, item, that = this,
                editData = that._editData;
            that.update(changeType);
            for (i = 0; i < editData.length; i++) {
                key = editData[i].key;
                item = that._generateNewItem(key);
                if (editData[i].type === DATA_EDIT_DATA_INSERT_TYPE && that._needInsertItem(editData[i], changeType, items, item)) {
                    items.splice(key.dataRowIndex, 0, item)
                }
            }
            return items
        },
        processDataItem: function(item, options, generateDataValues) {
            var data, editMode, editData, editIndex, that = this,
                columns = options.visibleColumns,
                key = item.data[INSERT_INDEX] ? item.data.key : item.key;
            editIndex = getIndexByKey(key, that._editData);
            if (editIndex >= 0) {
                editMode = getEditMode(that);
                editData = that._editData[editIndex];
                data = editData.data;
                item.isEditing = options.rowIndex === that._getVisibleEditRowIndex();
                switch (editData.type) {
                    case DATA_EDIT_DATA_INSERT_TYPE:
                        if (editMode === EDIT_MODE_POPUP) {
                            item.visible = false
                        }
                        item.inserted = true;
                        item.key = key;
                        item.data = data;
                        break;
                    case DATA_EDIT_DATA_UPDATE_TYPE:
                        item.modified = true;
                        item.oldData = item.data;
                        item.data = deepExtendArraySafe(deepExtendArraySafe({}, item.data), data);
                        item.modifiedValues = generateDataValues(data, columns);
                        break;
                    case DATA_EDIT_DATA_REMOVE_TYPE:
                        if (editMode === EDIT_MODE_BATCH) {
                            item.data = deepExtendArraySafe(deepExtendArraySafe({}, item.data), data)
                        }
                        item.removed = true
                }
            }
        },
        insertRow: function() {
            errors.log("W0002", "dxDataGrid", "insertRow", "15.2", "Use the 'addRow' method instead");
            return this.addRow()
        },
        _initNewRow: function(options, insertKey) {
            this.executeAction("onInitNewRow", options);
            var rows = this._dataController.items(),
                row = rows[insertKey.rowIndex];
            if (row && (!row.isEditing && "detail" === row.rowType || "detailAdaptive" === row.rowType)) {
                insertKey.rowIndex++
            }
            insertKey.dataRowIndex = rows.filter(function(row, index) {
                return index < insertKey.rowIndex && "data" === row.rowType
            }).length
        },
        _getInsertIndex: function() {
            var maxInsertIndex = 0;
            this._editData.forEach(function(editItem) {
                if (editItem.type === DATA_EDIT_DATA_INSERT_TYPE && editItem.key[INSERT_INDEX] > maxInsertIndex) {
                    maxInsertIndex = editItem.key[INSERT_INDEX]
                }
            });
            return maxInsertIndex + 1
        },
        addRow: function(parentKey) {
            var $firstCell, that = this,
                dataController = that._dataController,
                store = dataController.store(),
                key = store && store.key(),
                rowsView = that.getView("rowsView"),
                param = {
                    data: {}
                },
                parentRowIndex = dataController.getRowIndexByKey(parentKey),
                insertKey = {
                    pageIndex: dataController.pageIndex(),
                    rowIndex: parentRowIndex >= 0 ? parentRowIndex + 1 : rowsView ? rowsView.getTopVisibleItemIndex() : 0,
                    parentKey: parentKey
                },
                oldEditRowIndex = that._getVisibleEditRowIndex(),
                editMode = getEditMode(that);
            if (editMode === EDIT_MODE_CELL && that.hasChanges()) {
                that.saveEditData()
            }
            that.refresh();
            var insertIndex = that._getInsertIndex();
            if (editMode !== EDIT_MODE_BATCH && insertIndex > 1) {
                return
            }
            if (!key) {
                param.data.__KEY__ = String(new Guid)
            }
            that._initNewRow(param, insertKey);
            if (editMode !== EDIT_MODE_BATCH) {
                that._editRowIndex = insertKey.rowIndex + that._dataController.getRowIndexOffset()
            }
            insertKey[INSERT_INDEX] = insertIndex;
            that._addEditData({
                key: insertKey,
                data: param.data,
                type: DATA_EDIT_DATA_INSERT_TYPE
            });
            dataController.updateItems({
                changeType: "update",
                rowIndices: [oldEditRowIndex, insertKey.rowIndex]
            });
            if (editMode === EDIT_MODE_POPUP) {
                that._showEditPopup(insertKey.rowIndex)
            } else {
                $firstCell = that.getFirstEditableCellInRow(insertKey.rowIndex);
                that._editCellInProgress = true;
                that._delayedInputFocus($firstCell, function() {
                    that._editCellInProgress = false;
                    var $cell = that.getFirstEditableCellInRow(insertKey.rowIndex);
                    $cell && $cell.trigger(clickEvent.name)
                })
            }
            that._afterInsertRow({
                key: insertKey,
                data: param.data
            })
        },
        _isEditingStart: function(options) {
            this.executeAction("onEditingStart", options);
            return options.cancel
        },
        _beforeEditCell: function(rowIndex, columnIndex, item) {
            if (getEditMode(this) === EDIT_MODE_CELL && !item.inserted && this.hasChanges()) {
                this.saveEditData();
                if (this.hasChanges()) {
                    return true
                }
            }
        },
        _beforeUpdateItems: function() {},
        _getVisibleEditRowIndex: function() {
            return this._editRowIndex >= 0 ? this._editRowIndex - this._dataController.getRowIndexOffset() : -1
        },
        editRow: function(rowIndex) {
            var $editingCell, that = this,
                dataController = that._dataController,
                items = dataController.items(),
                item = items[rowIndex],
                params = {
                    data: item.data,
                    cancel: false
                },
                oldEditRowIndex = that._getVisibleEditRowIndex();
            if (rowIndex === oldEditRowIndex) {
                return true
            }
            if (!item.inserted) {
                params.key = item.key
            }
            if (that._isEditingStart(params)) {
                return
            }
            that.init();
            that._pageIndex = dataController.pageIndex();
            that._editRowIndex = (items[0].inserted ? rowIndex - 1 : rowIndex) + that._dataController.getRowIndexOffset();
            that._addEditData({
                data: {},
                key: item.key,
                oldData: item.data
            });
            var rowIndices = [oldEditRowIndex, rowIndex],
                editMode = getEditMode(that);
            that._beforeUpdateItems(rowIndices, rowIndex, oldEditRowIndex);
            if (editMode === EDIT_MODE_POPUP) {
                that._showEditPopup(rowIndex)
            } else {
                dataController.updateItems({
                    changeType: "update",
                    rowIndices: rowIndices
                })
            }
            if (MODES_WITH_DELAYED_FOCUS.indexOf(editMode) !== -1) {
                $editingCell = that.getFocusedCellInRow(that._getVisibleEditRowIndex());
                that._delayedInputFocus($editingCell, function() {
                    $editingCell && that.component.focus($editingCell)
                })
            }
        },
        _showEditPopup: function(rowIndex) {
            var that = this,
                isMobileDevice = "desktop" !== devices.current().deviceType,
                popupOptions = extend({
                    showTitle: false,
                    fullScreen: isMobileDevice,
                    toolbarItems: [{
                        toolbar: "bottom",
                        location: "after",
                        widget: "dxButton",
                        options: that._getSaveButtonConfig()
                    }, {
                        toolbar: "bottom",
                        location: "after",
                        widget: "dxButton",
                        options: that._getCancelButtonConfig()
                    }],
                    contentTemplate: that._getPopupEditFormTemplate(rowIndex)
                }, that.option("editing.popup"));
            if (!that._editPopup) {
                var $popupContainer = $("<div>").appendTo(that.component.element()).addClass(that.addWidgetPrefix(EDIT_POPUP_CLASS));
                that._editPopup = that._createComponent($popupContainer, Popup, {});
                that._editPopup.on("hidden", that._getEditPopupHiddenHandler());
                that._editPopup.on("shown", function(e) {
                    e.component.content().find(FOCUSABLE_ELEMENT_SELECTOR).first().focus()
                })
            }
            that._editPopup.option(popupOptions);
            that._editPopup.show()
        },
        _getEditPopupHiddenHandler: function() {
            var that = this;
            return function(e) {
                if (that.isEditing()) {
                    that.cancelEditData()
                }
            }
        },
        _getPopupEditFormTemplate: function(rowIndex) {
            var that = this,
                rowData = that.component.getVisibleRows()[rowIndex],
                templateOptions = {
                    row: rowData,
                    rowType: rowData.rowType,
                    key: rowData.key
                };
            return function($container) {
                var formTemplate = that.getEditFormTemplate();
                formTemplate($container, templateOptions, true)
            }
        },
        _getSaveButtonConfig: function() {
            return {
                text: this.option("editing.texts.saveRowChanges"),
                onClick: this.saveEditData.bind(this)
            }
        },
        _getCancelButtonConfig: function() {
            return {
                text: this.option("editing.texts.cancelRowChanges"),
                onClick: this.cancelEditData.bind(this)
            }
        },
        editCell: function(rowIndex, columnIndex) {
            var $cell, showEditorAlways, that = this,
                columnsController = that._columnsController,
                dataController = that._dataController,
                items = dataController.items(),
                item = items[rowIndex],
                params = {
                    data: item && item.data,
                    cancel: false
                },
                oldEditRowIndex = that._getVisibleEditRowIndex(),
                oldEditColumnIndex = that._editColumnIndex,
                columns = columnsController.getVisibleColumns(),
                rowsView = that.getView("rowsView");
            if (commonUtils.isString(columnIndex)) {
                columnIndex = columnsController.columnOption(columnIndex, "index");
                columnIndex = columnsController.getVisibleIndex(columnIndex)
            }
            params.column = columnsController.getVisibleColumns()[columnIndex];
            showEditorAlways = params.column && params.column.showEditorAlways;
            if (params.column && item && ("data" === item.rowType || "detailAdaptive" === item.rowType) && !item.removed && !isRowEditMode(that)) {
                if (this.isEditCell(rowIndex, columnIndex)) {
                    return true
                }
                var editRowIndex = rowIndex + that._dataController.getRowIndexOffset();
                if (that._beforeEditCell(rowIndex, columnIndex, item)) {
                    return true
                }
                if (!item.inserted) {
                    params.key = item.key
                }
                if (that._isEditingStart(params)) {
                    return true
                }
                that._editRowIndex = editRowIndex;
                that._editColumnIndex = columnIndex;
                that._pageIndex = dataController.pageIndex();
                that._addEditData({
                    data: {},
                    key: item.key,
                    oldData: item.data
                });
                if (!showEditorAlways || columns[oldEditColumnIndex] && !columns[oldEditColumnIndex].showEditorAlways) {
                    that._editCellInProgress = true;
                    that.getController("editorFactory").loseFocus();
                    dataController.updateItems({
                        changeType: "update",
                        rowIndices: [oldEditRowIndex, that._getVisibleEditRowIndex()]
                    })
                }
                $cell = rowsView && rowsView.getCellElement(that._getVisibleEditRowIndex(), that._editColumnIndex);
                if ($cell && !$cell.find(":focus").length) {
                    that._focusEditingCell(function() {
                        that._editCellInProgress = false
                    }, $cell, true)
                } else {
                    that._editCellInProgress = false
                }
                return true
            }
            return false
        },
        _delayedInputFocus: function($cell, beforeFocusCallback, callBeforeFocusCallbackAlways) {
            var that = this;

            function inputFocus() {
                if (beforeFocusCallback) {
                    beforeFocusCallback()
                }
                $cell && $cell.find(FOCUSABLE_ELEMENT_SELECTOR).first().focus();
                that._beforeFocusCallback = null
            }
            if (devices.real().ios || devices.real().android) {
                inputFocus()
            } else {
                if (that._beforeFocusCallback) {
                    that._beforeFocusCallback()
                }
                clearTimeout(that._inputFocusTimeoutID);
                if (callBeforeFocusCallbackAlways) {
                    that._beforeFocusCallback = beforeFocusCallback
                }
                that._inputFocusTimeoutID = setTimeout(inputFocus)
            }
        },
        _focusEditingCell: function(beforeFocusCallback, $editCell, callBeforeFocusCallbackAlways) {
            var that = this,
                rowsView = that.getView("rowsView");
            $editCell = $editCell || rowsView && rowsView.getCellElement(that._getVisibleEditRowIndex(), that._editColumnIndex);
            that._delayedInputFocus($editCell, beforeFocusCallback, callBeforeFocusCallbackAlways)
        },
        removeRow: function(rowIndex) {
            errors.log("W0002", "dxDataGrid", "removeRow", "15.2", "Use the 'deleteRow' method instead");
            return this.deleteRow(rowIndex)
        },
        deleteRow: function(rowIndex) {
            var removeByKey, showDialogTitle, that = this,
                editingOptions = that.option("editing"),
                editingTexts = editingOptions && editingOptions.texts,
                confirmDeleteTitle = editingTexts && editingTexts.confirmDeleteTitle,
                isBatchMode = editingOptions && editingOptions.mode === EDIT_MODE_BATCH,
                confirmDeleteMessage = editingTexts && editingTexts.confirmDeleteMessage,
                dataController = that._dataController,
                oldEditRowIndex = that._getVisibleEditRowIndex(),
                item = dataController.items()[rowIndex],
                key = item && item.key;
            if (item) {
                removeByKey = function(key) {
                    that.refresh();
                    var editIndex = getIndexByKey(key, that._editData);
                    if (editIndex >= 0) {
                        if (that._editData[editIndex].type === DATA_EDIT_DATA_INSERT_TYPE) {
                            that._editData.splice(editIndex, 1)
                        } else {
                            that._editData[editIndex].type = DATA_EDIT_DATA_REMOVE_TYPE
                        }
                    } else {
                        that._addEditData({
                            key: key,
                            oldData: item.data,
                            type: DATA_EDIT_DATA_REMOVE_TYPE
                        })
                    }
                    if (isBatchMode) {
                        dataController.updateItems({
                            changeType: "update",
                            rowIndices: [oldEditRowIndex, rowIndex]
                        })
                    } else {
                        that.saveEditData()
                    }
                };
                if (isBatchMode || !confirmDeleteMessage) {
                    removeByKey(key)
                } else {
                    showDialogTitle = commonUtils.isDefined(confirmDeleteTitle) && confirmDeleteTitle.length > 0;
                    dialog.confirm(confirmDeleteMessage, confirmDeleteTitle, showDialogTitle).done(function(confirmResult) {
                        if (confirmResult) {
                            removeByKey(key)
                        }
                    })
                }
            }
        },
        undeleteRow: function(rowIndex) {
            var that = this,
                dataController = that._dataController,
                item = dataController.items()[rowIndex],
                oldEditRowIndex = that._getVisibleEditRowIndex(),
                key = item && item.key;
            if (item) {
                var editData, editIndex = getIndexByKey(key, that._editData);
                if (editIndex >= 0) {
                    editData = that._editData[editIndex];
                    if (typeUtils.isEmptyObject(editData.data)) {
                        that._editData.splice(editIndex, 1)
                    } else {
                        editData.type = DATA_EDIT_DATA_UPDATE_TYPE
                    }
                    dataController.updateItems({
                        changeType: "update",
                        rowIndices: [oldEditRowIndex, rowIndex]
                    })
                }
            }
        },
        _saveEditDataCore: function(deferreds, results) {
            var that = this,
                store = that._dataController.store(),
                isDataSaved = true;

            function executeEditingAction(actionName, params, func) {
                var deferred = $.Deferred();
                that.executeAction(actionName, params);

                function createFailureHandler(deferred) {
                    return function(arg) {
                        var error = arg instanceof Error ? arg : new Error(arg && String(arg) || "Unknown error");
                        deferred.reject(error)
                    }
                }
                when(params.cancel).done(function(cancel) {
                    if (cancel) {
                        deferred.resolve("cancel")
                    } else {
                        func(params).done(deferred.resolve).fail(createFailureHandler(deferred))
                    }
                }).fail(createFailureHandler(deferred));
                return deferred
            }
            $.each(that._editData, function(index, editData) {
                var deferred, doneDeferred, params, data = editData.data,
                    oldData = editData.oldData,
                    type = editData.type;
                if (that._beforeSaveEditData(editData, index)) {
                    return
                }
                switch (type) {
                    case DATA_EDIT_DATA_REMOVE_TYPE:
                        params = {
                            data: oldData,
                            key: editData.key,
                            cancel: false
                        };
                        deferred = executeEditingAction("onRowRemoving", params, function() {
                            return store.remove(editData.key)
                        });
                        break;
                    case DATA_EDIT_DATA_INSERT_TYPE:
                        params = {
                            data: data,
                            cancel: false
                        };
                        deferred = executeEditingAction("onRowInserting", params, function() {
                            return store.insert(params.data).done(function(data, key) {
                                editData.key = key
                            })
                        });
                        break;
                    case DATA_EDIT_DATA_UPDATE_TYPE:
                        params = {
                            newData: data,
                            oldData: oldData,
                            key: editData.key,
                            cancel: false
                        };
                        deferred = executeEditingAction("onRowUpdating", params, function() {
                            return store.update(editData.key, params.newData)
                        })
                }
                if (deferred) {
                    doneDeferred = $.Deferred();
                    deferred.always(function(data) {
                        isDataSaved = "cancel" !== data;
                        results.push({
                            key: editData.key,
                            result: data
                        })
                    }).always(doneDeferred.resolve);
                    deferreds.push(doneDeferred.promise())
                }
            });
            return isDataSaved
        },
        _processSaveEditDataResult: function(results) {
            var i, arg, cancel, editData, editIndex, isError, $popupContent, that = this,
                dataController = that._dataController,
                hasSavedData = false,
                editMode = getEditMode(that);
            for (i = 0; i < results.length; i++) {
                arg = results[i].result;
                cancel = "cancel" === arg;
                editIndex = getIndexByKey(results[i].key, that._editData);
                editData = that._editData[editIndex];
                if (editData) {
                    isError = arg && arg instanceof Error;
                    if (isError) {
                        editData.error = arg;
                        $popupContent = that.getPopupContent();
                        dataController.dataErrorOccurred.fire(arg, $popupContent);
                        if (editMode !== EDIT_MODE_BATCH) {
                            break
                        }
                    } else {
                        if (!cancel || editMode !== EDIT_MODE_BATCH && editData.type === DATA_EDIT_DATA_REMOVE_TYPE) {
                            that._editData.splice(editIndex, 1);
                            hasSavedData = !cancel
                        }
                    }
                }
            }
            return hasSavedData
        },
        _fireSaveEditDataEvents: function(editData) {
            var that = this;
            $.each(editData, function(_, itemData) {
                var data = itemData.data,
                    key = itemData.key,
                    type = itemData.type,
                    params = {
                        key: key,
                        data: data
                    };
                if (itemData.error) {
                    params.error = itemData.error
                }
                switch (type) {
                    case DATA_EDIT_DATA_REMOVE_TYPE:
                        that.executeAction("onRowRemoved", extend({}, params, {
                            data: itemData.oldData
                        }));
                        break;
                    case DATA_EDIT_DATA_INSERT_TYPE:
                        that.executeAction("onRowInserted", params);
                        break;
                    case DATA_EDIT_DATA_UPDATE_TYPE:
                        that.executeAction("onRowUpdated", params)
                }
            })
        },
        saveEditData: function() {
            var editData, that = this,
                results = [],
                deferreds = [],
                dataController = that._dataController,
                dataSource = dataController.dataSource(),
                editMode = getEditMode(that),
                result = $.Deferred();
            var resetEditIndices = function(that) {
                if (editMode !== EDIT_MODE_CELL) {
                    that._editColumnIndex = -1;
                    that._editRowIndex = -1
                }
            };
            if (that._beforeSaveEditData() || that._saving) {
                that._afterSaveEditData();
                return result.resolve().promise()
            }
            if (!that._saveEditDataCore(deferreds, results) && editMode === EDIT_MODE_CELL) {
                that._focusEditingCell()
            }
            if (deferreds.length) {
                that._saving = true;
                dataSource && dataSource.beginLoading();
                when.apply($, deferreds).done(function() {
                    editData = that._editData.slice(0);
                    if (that._processSaveEditDataResult(results)) {
                        resetEditIndices(that);
                        if (editMode === EDIT_MODE_POPUP && that._editPopup) {
                            that._editPopup.hide()
                        }
                        dataSource && dataSource.endLoading();
                        when(dataController.refresh()).always(function() {
                            that._fireSaveEditDataEvents(editData);
                            that._afterSaveEditData();
                            that._focusEditingCell();
                            result.resolve()
                        })
                    } else {
                        dataSource && dataSource.endLoading();
                        result.resolve()
                    }
                }).fail(function() {
                    dataSource && dataSource.endLoading();
                    result.resolve()
                });
                return result.always(function() {
                    that._saving = false
                }).promise()
            }
            if (isRowEditMode(that)) {
                if (!that.hasChanges()) {
                    that.cancelEditData()
                }
            } else {
                if (CELL_BASED_MODES.indexOf(editMode) !== -1) {
                    resetEditIndices(that);
                    dataController.updateItems()
                } else {
                    that._focusEditingCell()
                }
            }
            that._afterSaveEditData();
            return result.resolve().promise()
        },
        _updateEditColumn: function() {
            var that = this,
                isEditColumnVisible = that._isEditColumnVisible();
            that._columnsController.addCommandColumn({
                command: "edit",
                visible: isEditColumnVisible,
                cssClass: "dx-command-edit",
                width: "auto"
            });
            that._columnsController.columnOption("command:edit", "visible", isEditColumnVisible)
        },
        _isEditColumnVisible: function() {
            var that = this,
                editingOptions = that.option("editing");
            if (editingOptions) {
                var editMode = getEditMode(that),
                    isVisibleWithCurrentEditMode = false;
                switch (editMode) {
                    case EDIT_MODE_ROW:
                        isVisibleWithCurrentEditMode = editingOptions.allowUpdating || editingOptions.allowAdding;
                        break;
                    case EDIT_MODE_FORM:
                    case EDIT_MODE_POPUP:
                        isVisibleWithCurrentEditMode = editingOptions.allowUpdating
                }
                return editingOptions.allowDeleting || isVisibleWithCurrentEditMode
            }
        },
        _updateEditButtons: function() {
            var that = this,
                headerPanel = that.getView("headerPanel"),
                hasChanges = that.hasChanges();
            if (headerPanel) {
                headerPanel.setToolbarItemDisabled("saveButton", !hasChanges);
                headerPanel.setToolbarItemDisabled("revertButton", !hasChanges)
            }
        },
        _applyModified: function($element) {
            $element && $element.addClass(CELL_MODIFIED)
        },
        _beforeCloseEditCellInBatchMode: function() {},
        cancelEditData: function() {
            var that = this,
                editMode = getEditMode(that),
                rowIndex = this._editRowIndex,
                dataController = that._dataController;
            that._beforeCancelEditData();
            that.init();
            if (ROW_BASED_MODES.indexOf(editMode) !== -1 && rowIndex >= 0) {
                dataController.updateItems({
                    changeType: "update",
                    rowIndices: [rowIndex, rowIndex + 1]
                })
            } else {
                dataController.updateItems()
            }
            if (editMode === EDIT_MODE_POPUP) {
                that._hideEditPopup()
            }
        },
        _hideEditPopup: function() {
            this._editPopup && this._editPopup.option("visible", false)
        },
        hasEditData: function() {
            return this.hasChanges()
        },
        closeEditCell: function() {
            var that = this,
                editMode = getEditMode(that),
                oldEditRowIndex = that._getVisibleEditRowIndex(),
                dataController = that._dataController;
            if (!isRowEditMode(that)) {
                setTimeout(function() {
                    if (editMode === EDIT_MODE_CELL && that.hasChanges()) {
                        that._editRowIndex = -1;
                        that._editColumnIndex = -1;
                        that.saveEditData()
                    } else {
                        if (oldEditRowIndex >= 0) {
                            var rowIndices = [oldEditRowIndex];
                            that._editRowIndex = -1;
                            that._editColumnIndex = -1;
                            that._beforeCloseEditCellInBatchMode(rowIndices);
                            dataController.updateItems({
                                changeType: "update",
                                rowIndices: rowIndices
                            })
                        }
                    }
                })
            }
        },
        update: function(changeType) {
            var that = this,
                dataController = that._dataController;
            if (dataController && that._pageIndex !== dataController.pageIndex()) {
                if ("refresh" === changeType) {
                    that.refresh()
                }
                that._pageIndex = dataController.pageIndex()
            }
            that._updateEditButtons()
        },
        _getRowIndicesForCascadeUpdating: function(row) {
            return [row.rowIndex]
        },
        updateFieldValue: function(options, value, text, forceUpdateRow) {
            var params, that = this,
                data = {},
                rowKey = options.key,
                $cellElement = options.cellElement,
                editMode = getEditMode(that);
            if (void 0 === rowKey) {
                that._dataController.dataErrorOccurred.fire(errors.Error("E1043"))
            }
            if (void 0 !== rowKey && options.column.setCellValue) {
                if (editMode === EDIT_MODE_BATCH) {
                    that._applyModified($cellElement, options)
                }
                options.value = value;
                options.column.setCellValue(data, value, text);
                if (text && options.column.displayValueMap) {
                    options.column.displayValueMap[value] = text
                }
                params = {
                    data: data,
                    key: rowKey,
                    oldData: options.data,
                    type: DATA_EDIT_DATA_UPDATE_TYPE
                };
                that._addEditData(params, options.row);
                that._updateEditButtons();
                if (options.column.showEditorAlways && getEditMode(that) === EDIT_MODE_CELL && options.row && !options.row.inserted) {
                    that.saveEditData()
                } else {
                    if (options.row && (forceUpdateRow || options.column.setCellValue !== options.column.defaultSetCellValue)) {
                        that._updateEditRow(options.row, forceUpdateRow)
                    }
                }
            }
        },
        _updateEditRow: function(row, forceUpdateRow) {
            var that = this,
                editMode = getEditMode(that);
            if (editMode === EDIT_MODE_POPUP) {
                setTimeout(this._updatePopupForm.bind(this, forceUpdateRow))
            } else {
                this._dataController.updateItems({
                    changeType: "update",
                    rowIndices: this._getRowIndicesForCascadeUpdating(row)
                });
                if (!forceUpdateRow) {
                    this._focusEditingCell()
                }
            }
        },
        _updatePopupForm: function(forceUpdateRow) {
            var columnIndex, $focusedItemElement, rowsView = this._rowsView,
                rowIndex = this.getEditFormRowIndex();
            if (rowIndex >= 0 && this._editForm) {
                if (!forceUpdateRow) {
                    $focusedItemElement = this._editForm.element().find(".dx-state-focused");
                    columnIndex = rowsView.getCellIndex($focusedItemElement, rowIndex)
                }
                this._editForm.repaint();
                if (columnIndex >= 0) {
                    $focusedItemElement = rowsView.getCellElement(rowIndex, columnIndex);
                    this._delayedInputFocus($focusedItemElement)
                }
            }
        },
        _addEditData: function(options, row) {
            var that = this,
                editDataIndex = getIndexByKey(options.key, that._editData);
            if (editDataIndex < 0) {
                editDataIndex = that._editData.length;
                that._editData.push(options)
            }
            if (that._editData[editDataIndex]) {
                options.type = that._editData[editDataIndex].type || options.type;
                deepExtendArraySafe(that._editData[editDataIndex], {
                    data: options.data,
                    type: options.type
                });
                if (row) {
                    row.data = deepExtendArraySafe(deepExtendArraySafe({}, row.data), options.data)
                }
            }
            return editDataIndex
        },
        _getFormEditItemTemplate: function(cellOptions, column) {
            return column.editCellTemplate || getDefaultEditorTemplate(this)
        },
        renderFormEditTemplate: function(detailCellOptions, item, form, $container, isReadOnly) {
            var that = this,
                column = item.column,
                rowData = detailCellOptions.row && detailCellOptions.row.data,
                cellOptions = extend({}, detailCellOptions, {
                    cellElement: null,
                    isOnForm: true,
                    item: item,
                    value: column.calculateCellValue(rowData),
                    column: extend({}, column, {
                        editorOptions: item.editorOptions
                    }),
                    id: form.getItemID(item.name || item.dataField),
                    columnIndex: column.index,
                    setValue: !isReadOnly && column.allowEditing && function(value) {
                        that.updateFieldValue(cellOptions, value)
                    }
                }),
                template = that._getFormEditItemTemplate.bind(that)(cellOptions, column);
            if (that._rowsView.renderTemplate($container, template, cellOptions, !!$container.closest(document).length)) {
                that._rowsView._updateCell($container, cellOptions)
            }
        },
        getFormEditorTemplate: function(cellOptions, item) {
            var that = this;
            return function(options, $container) {
                that.renderFormEditTemplate.bind(that)(cellOptions, item, options.component, $container)
            }
        },
        getEditFormTemplate: function() {
            var that = this;
            return function($container, detailOptions, renderFormOnly) {
                var editFormOptions = that.option("editing.form"),
                    items = that.option("editing.form.items"),
                    userCustomizeItem = that.option("editing.form.customizeItem"),
                    editData = that._editData[getIndexByKey(detailOptions.key, that._editData)],
                    editFormItemClass = that.addWidgetPrefix(EDIT_FORM_ITEM_CLASS),
                    isScrollingEnabled = getEditMode(that) === EDIT_MODE_POPUP;
                if (!items) {
                    var columns = that.getController("columns").getColumns();
                    items = [];
                    $.each(columns, function(_, column) {
                        if (!column.isBand) {
                            items.push({
                                column: column,
                                name: column.name,
                                dataField: column.dataField
                            })
                        }
                    })
                }
                that._firstFormItem = void 0;
                that._editForm = that._createComponent($("<div>").appendTo($container), Form, extend({
                    scrollingEnabled: isScrollingEnabled
                }, editFormOptions, {
                    items: items,
                    formID: "dx-" + new Guid,
                    validationGroup: editData,
                    customizeItem: function(item) {
                        var column = item.column || that._columnsController.columnOption(item.name || item.dataField);
                        if (column) {
                            item.label = item.label || {};
                            item.label.text = item.label.text || column.caption;
                            item.template = item.template || that.getFormEditorTemplate(detailOptions, item);
                            item.column = column;
                            if (column.formItem) {
                                extend(item, column.formItem)
                            }
                            var itemVisible = commonUtils.isDefined(item.visible) ? item.visible : true;
                            if (!that._firstFormItem && itemVisible) {
                                that._firstFormItem = item
                            }
                        }
                        userCustomizeItem && userCustomizeItem.call(this, item);
                        item.cssClass = commonUtils.isString(item.cssClass) ? item.cssClass + " " + editFormItemClass : editFormItemClass
                    }
                }));
                if (!renderFormOnly) {
                    var $buttonsContainer = $("<div>").addClass(that.addWidgetPrefix(FORM_BUTTONS_CONTAINER_CLASS)).appendTo($container);
                    that._createComponent($("<div>").appendTo($buttonsContainer), Button, that._getSaveButtonConfig());
                    that._createComponent($("<div>").appendTo($buttonsContainer), Button, that._getCancelButtonConfig())
                }
            }
        },
        getColumnTemplate: function(options) {
            var template, editingOptions, editingTexts, allowUpdating, editingStartOptions, that = this,
                column = options.column,
                rowIndex = options.row && options.row.rowIndex,
                isRowMode = isRowEditMode(that),
                isRowEditing = that.isEditRow(rowIndex),
                isCellEditing = that.isEditCell(rowIndex, options.columnIndex);
            if ((column.showEditorAlways || column.setCellValue && (isRowEditing && column.allowEditing || isCellEditing)) && ("data" === options.rowType || "detailAdaptive" === options.rowType) && !column.command) {
                allowUpdating = that.option("editing.allowUpdating");
                if (((allowUpdating || isRowEditing) && column.allowEditing || isCellEditing) && (isRowMode && isRowEditing || !isRowMode)) {
                    if (column.showEditorAlways && !isRowMode) {
                        editingStartOptions = {
                            cancel: false,
                            key: options.row.inserted ? void 0 : options.row.key,
                            data: options.row.data,
                            column: column
                        };
                        that._isEditingStart(editingStartOptions)
                    }
                    if (!editingStartOptions || !editingStartOptions.cancel) {
                        options.setValue = function(value, text) {
                            that.updateFieldValue(options, value, text)
                        }
                    }
                }
                template = column.editCellTemplate || getDefaultEditorTemplate(that)
            } else {
                if ("edit" === column.command && "data" === options.rowType) {
                    template = function(container, options) {
                        container.css("text-align", "center");
                        options.rtlEnabled = that.option("rtlEnabled");
                        editingOptions = that.option("editing") || {};
                        editingTexts = editingOptions.texts || {};
                        if (options.row && options.row.rowIndex === that._getVisibleEditRowIndex() && isRowMode) {
                            that._createLink(container, editingTexts.saveRowChanges, "saveEditData", options, "dx-link-save");
                            that._createLink(container, editingTexts.cancelRowChanges, "cancelEditData", options, "dx-link-cancel")
                        } else {
                            that._createEditingLinks(container, options, editingOptions, isRowMode)
                        }
                    }
                } else {
                    if ("detail" === column.command && "detail" === options.rowType && isRowEditing) {
                        template = that.getEditFormTemplate(options)
                    }
                }
            }
            return template
        },
        _createLink: function(container, text, methodName, options, linkClass) {
            var that = this,
                $link = $("<a>").addClass(LINK_CLASS).addClass(linkClass).text(text).on(addNamespace(clickEvent.name, EDITING_NAMESPACE), that.createAction(function(params) {
                    var e = params.jQueryEvent;
                    e.stopPropagation();
                    setTimeout(function() {
                        options.row && that[methodName](options.row.rowIndex)
                    })
                }));
            options.rtlEnabled ? container.prepend($link, "&nbsp;") : container.append($link, "&nbsp;")
        },
        _createEditingLinks: function(container, options, editingOptions, isRowMode) {
            var editingTexts = editingOptions.texts || {};
            if (editingOptions.allowUpdating && isRowMode) {
                this._createLink(container, editingTexts.editRow, "editRow", options, "dx-link-edit")
            }
            if (editingOptions.allowDeleting) {
                if (options.row.removed) {
                    this._createLink(container, editingTexts.undeleteRow, "undeleteRow", options, "dx-link-undelete")
                } else {
                    this._createLink(container, editingTexts.deleteRow, "deleteRow", options, "dx-link-delete")
                }
            }
        },
        prepareEditButtons: function(headerPanel) {
            var that = this,
                editingOptions = that.option("editing") || {},
                editingTexts = that.option("editing.texts") || {},
                titleButtonTextByClassNames = {
                    revert: editingTexts.cancelAllChanges,
                    save: editingTexts.saveAllChanges,
                    addRow: editingTexts.addRow
                },
                classNameButtonByNames = {
                    revert: "cancel",
                    save: "save",
                    addRow: "addrow"
                },
                buttonItems = [];
            var prepareButtonItem = function(name, methodName, sortIndex) {
                var className = classNameButtonByNames[name],
                    onInitialized = function(e) {
                        e.element.addClass(headerPanel._getToolbarButtonClass(EDIT_BUTTON_CLASS + " " + that.addWidgetPrefix(className) + "-button"))
                    },
                    hintText = titleButtonTextByClassNames[name],
                    isButtonDisabled = ("save" === className || "cancel" === className) && !that.hasChanges();
                return {
                    widget: "dxButton",
                    options: {
                        onInitialized: onInitialized,
                        icon: "edit-button-" + className,
                        disabled: isButtonDisabled,
                        onClick: function() {
                            that[methodName]()
                        },
                        text: hintText,
                        hint: hintText
                    },
                    showText: "inMenu",
                    name: name + "Button",
                    location: "after",
                    locateInMenu: "auto",
                    sortIndex: sortIndex
                }
            };
            if (editingOptions.allowAdding) {
                buttonItems.push(prepareButtonItem("addRow", "addRow", 20))
            }
            if ((editingOptions.allowUpdating || editingOptions.allowAdding || editingOptions.allowDeleting) && getEditMode(that) === EDIT_MODE_BATCH) {
                buttonItems.push(prepareButtonItem("save", "saveEditData", 21));
                buttonItems.push(prepareButtonItem("revert", "cancelEditData", 22))
            }
            return buttonItems
        },
        showHighlighting: function($cell) {
            var $highlight = $cell.find("." + CELL_HIGHLIGHT_OUTLINE);
            if ("TD" === $cell.get(0).tagName && !$highlight.length) {
                $cell.wrapInner($("<div>").addClass(CELL_HIGHLIGHT_OUTLINE + " " + POINTER_EVENTS_TARGET_CLASS))
            }
        },
        resetRowAndPageIndices: function(alwaysRest) {
            var that = this;
            $.each(that._editData, function(_, editData) {
                if (editData.pageIndex !== that._pageIndex || alwaysRest) {
                    delete editData.pageIndex;
                    delete editData.rowIndex
                }
            })
        },
        _afterInsertRow: function() {},
        _beforeSaveEditData: function() {},
        _afterSaveEditData: function() {},
        _beforeCancelEditData: function() {}
    }
}());
module.exports = {
    defaultOptions: function() {
        return {
            editing: {
                mode: "row",
                allowAdding: false,
                allowUpdating: false,
                allowDeleting: false,
                texts: {
                    editRow: messageLocalization.format("dxDataGrid-editingEditRow"),
                    saveAllChanges: messageLocalization.format("dxDataGrid-editingSaveAllChanges"),
                    saveRowChanges: messageLocalization.format("dxDataGrid-editingSaveRowChanges"),
                    cancelAllChanges: messageLocalization.format("dxDataGrid-editingCancelAllChanges"),
                    cancelRowChanges: messageLocalization.format("dxDataGrid-editingCancelRowChanges"),
                    addRow: messageLocalization.format("dxDataGrid-editingAddRow"),
                    deleteRow: messageLocalization.format("dxDataGrid-editingDeleteRow"),
                    undeleteRow: messageLocalization.format("dxDataGrid-editingUndeleteRow"),
                    confirmDeleteMessage: messageLocalization.format("dxDataGrid-editingConfirmDeleteMessage"),
                    confirmDeleteTitle: ""
                },
                form: {
                    colCount: 2
                },
                popup: {}
            }
        }
    },
    controllers: {
        editing: EditingController
    },
    extenders: {
        controllers: {
            data: {
                init: function() {
                    this._editingController = this.getController("editing");
                    this.callBase()
                },
                reload: function(full) {
                    var d, editingController = this.getController("editing");
                    this._editingController.refresh();
                    d = this.callBase(full);
                    return d && d.done(function() {
                        editingController.resetRowAndPageIndices(true)
                    })
                },
                _updateItemsCore: function(change) {
                    this.callBase(change);
                    var editingController = this._editingController,
                        editFormRowIndex = editingController.getEditMode() === EDIT_MODE_FORM && editingController.getEditFormRowIndex(),
                        editFormItem = this.items()[editFormRowIndex];
                    if (editFormItem) {
                        editFormItem.rowType = "detail"
                    }
                },
                _processItems: function(items, changeType) {
                    items = this._editingController.processItems(items, changeType);
                    return this.callBase(items, changeType)
                },
                _processDataItem: function(dataItem, options) {
                    this._editingController.processDataItem(dataItem, options, this.generateDataValues);
                    return this.callBase(dataItem, options)
                },
                _processItem: function(item, options) {
                    item = this.callBase(item, options);
                    if (item.inserted) {
                        options.dataIndex--;
                        delete item.dataIndex
                    }
                    return item
                }
            }
        },
        views: {
            rowsView: {
                init: function() {
                    this.callBase();
                    this._editingController = this.getController("editing")
                },
                getCellElements: function(rowIndex) {
                    var $cellElements = this.callBase(rowIndex),
                        editingController = this._editingController,
                        editForm = editingController.getEditForm(),
                        editFormRowIndex = editingController.getEditFormRowIndex();
                    if (editFormRowIndex === rowIndex && $cellElements && editForm) {
                        return editForm.element().find("." + this.addWidgetPrefix(EDIT_FORM_ITEM_CLASS) + ", ." + BUTTON_CLASS)
                    }
                    return $cellElements
                },
                getCellIndex: function($cell, rowIndex) {
                    if (!$cell.is("td") && rowIndex >= 0) {
                        var $cellElements = this.getCellElements(rowIndex),
                            cellIndex = -1;
                        $.each($cellElements, function(index, cellElement) {
                            if ($(cellElement).find($cell).length) {
                                cellIndex = index;
                                return false
                            }
                        });
                        return cellIndex
                    }
                    return this.callBase.apply(this, arguments)
                },
                _getVisibleColumnIndex: function($cells, rowIndex, columnIdentifier) {
                    var item, visibleIndex = this.callBase($cells, rowIndex, columnIdentifier),
                        editFormRowIndex = this._editingController.getEditFormRowIndex();
                    if (editFormRowIndex === rowIndex) {
                        $.each($cells, function(index, cellElement) {
                            item = $(cellElement).find(".dx-field-item-content").data("dx-form-item");
                            if (item && item.column && item.column.visibleIndex === visibleIndex) {
                                visibleIndex = index;
                                return false
                            }
                        })
                    }
                    return visibleIndex
                },
                publicMethods: function() {
                    return this.callBase().concat(["cellValue"])
                },
                _getCellTemplate: function(options) {
                    var that = this,
                        template = that._editingController.getColumnTemplate(options);
                    return template || that.callBase(options)
                },
                _isNativeClick: function() {
                    return (devices.real().ios || devices.real().android) && this.option("editing.allowUpdating")
                },
                _createTable: function() {
                    var that = this,
                        $table = that.callBase.apply(that, arguments);
                    if (!isRowEditMode(that) && that.option("editing.allowUpdating")) {
                        $table.on(addNamespace(holdEvent.name, "dxDataGridRowsView"), "td:not(." + EDITOR_CELL_CLASS + ")", that.createAction(function() {
                            var editingController = that._editingController;
                            if (editingController.isEditing()) {
                                editingController.closeEditCell()
                            }
                        }))
                    }
                    return $table
                },
                _createRow: function(row) {
                    var editingController, isEditRow, isRowRemoved, isRowInserted, isRowModified, $row = this.callBase(row);
                    if (row) {
                        editingController = this._editingController;
                        isEditRow = editingController.isEditRow(row.rowIndex);
                        isRowRemoved = !!row.removed;
                        isRowInserted = !!row.inserted;
                        isRowModified = !!row.modified;
                        if (getEditMode(this) === EDIT_MODE_BATCH) {
                            isRowRemoved && $row.addClass(ROW_REMOVED)
                        } else {
                            isEditRow && $row.addClass(EDIT_ROW)
                        }
                        isRowInserted && $row.addClass(ROW_INSERTED);
                        isRowModified && $row.addClass(ROW_MODIFIED);
                        if (isEditRow || isRowInserted || isRowRemoved) {
                            $row.removeClass(ROW_SELECTED)
                        }
                        if (isEditRow && "detail" === row.rowType) {
                            $row.addClass(this.addWidgetPrefix(EDIT_FORM_CLASS))
                        }
                    }
                    return $row
                },
                _getColumnIndexByElement: function($element) {
                    var $tableElement = $element.closest("table"),
                        $tableElements = this.getTableElements();
                    while ($tableElement.length && !$tableElements.filter($tableElement).length) {
                        $element = $tableElement.closest("td");
                        $tableElement = $element.closest("table")
                    }
                    return this._getColumnIndexByElementCore($element)
                },
                _getColumnIndexByElementCore: function($element) {
                    var $targetElement = $element.closest("." + ROW_CLASS + "> td:not(.dx-master-detail-cell)");
                    return this.getCellIndex($targetElement)
                },
                _rowClick: function(e) {
                    var that = this,
                        editingController = that._editingController,
                        $targetElement = $(e.jQueryEvent.target),
                        columnIndex = that._getColumnIndexByElement($targetElement),
                        row = that._dataController.items()[e.rowIndex],
                        allowUpdating = that.option("editing.allowUpdating") || row && row.inserted,
                        column = that._columnsController.getVisibleColumns()[columnIndex],
                        allowEditing = column && (column.allowEditing || editingController.isEditCell(e.rowIndex, columnIndex));
                    if ($targetElement.closest("." + ROW_CLASS + "> td").hasClass(POINTER_EVENTS_NONE_CLASS)) {
                        return
                    }
                    if (!(allowUpdating && allowEditing && editingController.editCell(e.rowIndex, columnIndex)) && !editingController.isEditRow(e.rowIndex)) {
                        that.callBase(e)
                    }
                },
                _cellPrepared: function($cell, parameters) {
                    var columnIndex = parameters.columnIndex,
                        editingController = this._editingController,
                        isCommandCell = !!parameters.column.command,
                        isEditableCell = parameters.setValue,
                        isEditing = parameters.isEditing || editingController.isEditRow(parameters.rowIndex) && parameters.column.allowEditing;
                    if ("data" === parameters.rowType && !parameters.column.command && (isEditing || parameters.column.showEditorAlways)) {
                        var alignment = parameters.column.alignment;
                        $cell.addClass(EDITOR_CELL_CLASS).toggleClass(this.addWidgetPrefix(READONLY_CLASS), !isEditableCell).toggleClass(CELL_FOCUS_DISABLED_CLASS, !isEditableCell);
                        if (alignment) {
                            $cell.find(EDITORS_INPUT_SELECTOR).first().css("text-align", alignment)
                        }
                    }
                    var modifiedValues = parameters.row && (parameters.row.inserted ? parameters.row.values : parameters.row.modifiedValues);
                    if (modifiedValues && void 0 !== modifiedValues[columnIndex] && parameters.column && !isCommandCell && parameters.column.setCellValue) {
                        editingController.showHighlighting($cell);
                        $cell.addClass(CELL_MODIFIED)
                    } else {
                        if (isEditableCell) {
                            editingController.showHighlighting($cell, true)
                        }
                    }
                    this.callBase.apply(this, arguments)
                },
                _formItemPrepared: function() {},
                _isFormItem: function(parameters) {
                    var isDetailRow = "detail" === parameters.rowType || "detailAdaptive" === parameters.rowType,
                        isPopupEditing = "data" === parameters.rowType && "popup" === getEditMode(this);
                    return (isDetailRow || isPopupEditing) && parameters.item
                },
                _updateCell: function($cell, parameters) {
                    if (this._isFormItem(parameters)) {
                        this._formItemPrepared(parameters, $cell)
                    } else {
                        this.callBase($cell, parameters)
                    }
                },
                _update: function(change) {
                    this.callBase(change);
                    if ("updateSelection" === change.changeType) {
                        this.getTableElements().children("tbody").children("." + EDIT_ROW).removeClass(ROW_SELECTED)
                    }
                },
                _getCellOptions: function(options) {
                    var cellOptions = this.callBase(options);
                    cellOptions.isEditing = this._editingController.isEditCell(cellOptions.rowIndex, cellOptions.columnIndex);
                    return cellOptions
                },
                cellValue: function(rowIndex, columnIdentifier, value, text) {
                    var cellOptions = this.getCellOptions(rowIndex, columnIdentifier);
                    if (cellOptions) {
                        if (void 0 === value) {
                            return cellOptions.value
                        } else {
                            this._editingController.updateFieldValue(cellOptions, value, text, true)
                        }
                    }
                }
            },
            headerPanel: {
                _getToolbarItems: function() {
                    var items = this.callBase(),
                        editButtonItems = this.getController("editing").prepareEditButtons(this);
                    return editButtonItems.concat(items)
                },
                optionChanged: function(args) {
                    switch (args.name) {
                        case "editing":
                            this._invalidate();
                            this.callBase(args);
                            break;
                        default:
                            this.callBase(args)
                    }
                },
                isVisible: function() {
                    var that = this,
                        editingOptions = that.getController("editing").option("editing");
                    return that.callBase() || editingOptions && (editingOptions.allowAdding || (editingOptions.allowUpdating || editingOptions.allowDeleting) && editingOptions.mode === EDIT_MODE_BATCH)
                }
            }
        }
    }
};
