/**
 * DevExtreme (animation/translator.js)
 * Version: 17.1.6
 * Build date: Tue Sep 05 2017
 *
 * Copyright (c) 2012 - 2017 Developer Express Inc. ALL RIGHTS RESERVED
 * Read about DevExtreme licensing here: https://js.devexpress.com/Licensing/
 */
"use strict";
var $ = require("../core/renderer"),
    type = require("../core/utils/common").type;
var TRANSLATOR_DATA_KEY = "dxTranslator",
    TRANSFORM_MATRIX_REGEX = /matrix(3d)?\((.+?)\)/,
    TRANSLATE_REGEX = /translate(?:3d)?\((.+?)\)/;
var locate = function($element) {
    var translate = getTranslate($element);
    return {
        left: translate.x,
        top: translate.y
    }
};
var move = function($element, position) {
    var translate, left = position.left,
        top = position.top;
    if (void 0 === left) {
        translate = getTranslate($element);
        translate.y = top || 0
    } else {
        if (void 0 === top) {
            translate = getTranslate($element);
            translate.x = left || 0
        } else {
            translate = {
                x: left || 0,
                y: top || 0,
                z: 0
            };
            cacheTranslate($element, translate)
        }
    }
    $element.css({
        transform: getTranslateCss(translate)
    });
    if (isPercentValue(left) || isPercentValue(top)) {
        clearCache($element)
    }
};
var isPercentValue = function(value) {
    return "string" === type(value) && "%" === value[value.length - 1]
};
var getTranslate = function($element) {
    var result = $element.length ? $.data($element.get(0), TRANSLATOR_DATA_KEY) : null;
    if (!result) {
        var transformValue = $element.css("transform") || getTranslateCss({
                x: 0,
                y: 0
            }),
            matrix = transformValue.match(TRANSFORM_MATRIX_REGEX),
            is3D = matrix && matrix[1];
        if (matrix) {
            matrix = matrix[2].split(",");
            if ("3d" === is3D) {
                matrix = matrix.slice(12, 15)
            } else {
                matrix.push(0);
                matrix = matrix.slice(4, 7)
            }
        } else {
            matrix = [0, 0, 0]
        }
        result = {
            x: parseFloat(matrix[0]),
            y: parseFloat(matrix[1]),
            z: parseFloat(matrix[2])
        };
        cacheTranslate($element, result)
    }
    return result
};
var cacheTranslate = function($element, translate) {
    if ($element.length) {
        $.data($element.get(0), TRANSLATOR_DATA_KEY, translate)
    }
};
var clearCache = function($element) {
    if ($element.length) {
        $.removeData($element.get(0), TRANSLATOR_DATA_KEY)
    }
};
var resetPosition = function($element) {
    $element.css({
        left: 0,
        top: 0,
        transform: "none"
    });
    clearCache($element)
};
var parseTranslate = function(translateString) {
    var result = translateString.match(TRANSLATE_REGEX);
    if (!result || !result[1]) {
        return
    }
    result = result[1].split(",");
    result = {
        x: parseFloat(result[0]),
        y: parseFloat(result[1]),
        z: parseFloat(result[2])
    };
    return result
};
var getTranslateCss = function(translate) {
    translate.x = translate.x || 0;
    translate.y = translate.y || 0;
    var xValueString = isPercentValue(translate.x) ? translate.x : translate.x + "px";
    var yValueString = isPercentValue(translate.y) ? translate.y : translate.y + "px";
    return "translate(" + xValueString + ", " + yValueString + ")"
};
exports.move = move;
exports.locate = locate;
exports.clearCache = clearCache;
exports.parseTranslate = parseTranslate;
exports.getTranslate = getTranslate;
exports.getTranslateCss = getTranslateCss;
exports.resetPosition = resetPosition;
